00001 /* multi_audio.h */
00002 
00003 /* Interface description for drivers implementing studio-level audio I/O with */
00004 /* possible auxillary functions (transport, time code, etc). */
00005 /* Copyright Â© 1998-2001 Be Incorporated. All rights reserved. */
00006 
00007 #if !defined(_MULTI_AUDIO_H)
00008 #define _MULTI_AUDIO_H
00009 
00010 #include <Drivers.h>
00011 #include <Debug.h>
00012 #include <audio_base.h>
00013 
00014 #define B_MULTI_DRIVER_BASE (B_AUDIO_DRIVER_BASE+20)
00015 
00016 enum {  /* open() modes */
00017     /*  O_RDONLY is 0, O_WRONLY is 1, O_RDWR is 2   */
00018     B_MULTI_CONTROL = 3
00019 };
00020 
00021 /* ioctl codes */
00022 enum {
00023     /* multi_description */
00024     B_MULTI_GET_DESCRIPTION = B_MULTI_DRIVER_BASE,
00025     /* multi_event_handling */
00026     B_MULTI_GET_EVENT_INFO,
00027     B_MULTI_SET_EVENT_INFO,
00028     B_MULTI_GET_EVENT,
00029     /* multi_channel_enable */
00030     B_MULTI_GET_ENABLED_CHANNELS,
00031     B_MULTI_SET_ENABLED_CHANNELS,
00032     /* multi_format_info */
00033     B_MULTI_GET_GLOBAL_FORMAT,
00034     B_MULTI_SET_GLOBAL_FORMAT,      /* always sets for all channels, always implemented */
00035     /* multi_channel_formats */
00036     B_MULTI_GET_CHANNEL_FORMATS,
00037     B_MULTI_SET_CHANNEL_FORMATS,    /* only implemented if possible */
00038     /* multi_mix_value_info */
00039     B_MULTI_GET_MIX,
00040     B_MULTI_SET_MIX,
00041     /* multi_mix_channel_info */
00042     B_MULTI_LIST_MIX_CHANNELS,
00043     /* multi_mix_control_info */
00044     B_MULTI_LIST_MIX_CONTROLS,
00045     /* multi_mix_connection_info */
00046     B_MULTI_LIST_MIX_CONNECTIONS,
00047     /* multi_buffer_list */
00048     B_MULTI_GET_BUFFERS,            /* Fill out the struct for the first time; doesn't start anything. */
00049     B_MULTI_SET_BUFFERS,            /* Set what buffers to use, if the driver supports soft buffers. */
00050     /* bigtime_t */
00051     B_MULTI_SET_START_TIME,         /* When to actually start */
00052     /* multi_buffer_info */
00053     B_MULTI_BUFFER_EXCHANGE,        /* stop and go are derived from this being called */
00054     B_MULTI_BUFFER_FORCE_STOP,      /* force stop of playback */
00055     /* extension protocol */
00056     B_MULTI_LIST_EXTENSIONS,        /* get a list of supported extensions */
00057     B_MULTI_GET_EXTENSION,          /* get the value of an extension (or return error if not supported) */
00058     B_MULTI_SET_EXTENSION,          /* set the value of an extension */
00059     /* multi_mode_list */
00060     B_MULTI_LIST_MODES,             /* get a list of possible modes (multi_mode_list * arg) */
00061     B_MULTI_GET_MODE,               /* get the current mode (int32 * arg) */
00062     B_MULTI_SET_MODE                /* set a new mode (int32 * arg) */
00063 };
00064 
00065 /*  possible sample lock sources */
00066 #define B_MULTI_LOCK_INPUT_CHANNEL 0x0  /* lock_source_data is channel id */
00067 #define B_MULTI_LOCK_INTERNAL 0x1
00068 #define B_MULTI_LOCK_WORDCLOCK 0x2
00069 #define B_MULTI_LOCK_SUPERCLOCK 0x4
00070 #define B_MULTI_LOCK_LIGHTPIPE 0x8
00071 #define B_MULTI_LOCK_VIDEO 0x10         /* or blackburst */
00072 #define B_MULTI_LOCK_FIRST_CARD 0x20    /* if you have more than one card */
00073 #define B_MULTI_LOCK_MTC 0x40
00074 #define B_MULTI_LOCK_SPDIF 0x80
00075 
00076 /* possible timecode sources */
00077 #define B_MULTI_TIMECODE_MTC 0x1
00078 #define B_MULTI_TIMECODE_VTC 0x2
00079 #define B_MULTI_TIMECODE_SMPTE 0x4
00080 #define B_MULTI_TIMECODE_SUPERCLOCK 0x8
00081 #define B_MULTI_TIMECODE_FIREWIRE 0x10
00082 
00083 /* interface_flags values */
00084 /* Available functions on this device. */
00085 #define B_MULTI_INTERFACE_PLAYBACK 0x1
00086 #define B_MULTI_INTERFACE_RECORD 0x2
00087 #define B_MULTI_INTERFACE_TRANSPORT 0x4
00088 #define B_MULTI_INTERFACE_TIMECODE 0x8
00089 /* "Soft" buffers means you can change the pointer values and the driver will still be happy. */
00090 #define B_MULTI_INTERFACE_SOFT_PLAY_BUFFERS 0x10000
00091 #define B_MULTI_INTERFACE_SOFT_REC_BUFFERS 0x20000
00092 /* Whether the data stream is interrupted when changing channel enables. */
00093 #define B_MULTI_INTERFACE_CLICKS_WHEN_ENABLING_OUTPUTS 0x40000
00094 #define B_MULTI_INTERFACE_CLICKS_WHEN_ENABLING_INPUTS 0x80000
00095 
00096 #define B_CURRENT_INTERFACE_VERSION 0x4502
00097 #define B_MINIMUM_INTERFACE_VERSION 0x4502
00098 
00099 typedef struct multi_description multi_description;
00100 typedef struct multi_channel_info multi_channel_info;
00101 struct multi_description {
00102 
00103     size_t          info_size;          /* sizeof(multi_description) */
00104     uint32          interface_version;  /* current version of interface that's implemented */
00105     uint32          interface_minimum;  /* minimum version required to understand driver */
00106 
00107     char            friendly_name[32];  /* name displayed to user (C string) */
00108     char            vendor_info[32];    /* name used internally by vendor (C string) */
00109 
00110     int32           output_channel_count;
00111     int32           input_channel_count;
00112     int32           output_bus_channel_count;
00113     int32           input_bus_channel_count;
00114     int32           aux_bus_channel_count;
00115 
00116     int32           request_channel_count;      /* how many channel_infos are there */
00117     multi_channel_info *
00118                     channels;
00119 
00120     uint32          output_rates;
00121     uint32          input_rates;
00122     float           min_cvsr_rate;
00123     float           max_cvsr_rate;
00124     uint32          output_formats;
00125     uint32          input_formats;
00126     uint32          lock_sources;
00127     uint32          timecode_sources;
00128 
00129     uint32          interface_flags;
00130     bigtime_t       start_latency;      /* how much in advance driver needs SET_START_TIME */
00131 
00132     uint32          _reserved_[11];
00133 
00134     char            control_panel[64];  /* MIME type of control panel application */
00135 };
00136 
00137 #define B_CHANNEL_MONO_BUS 0x4000000
00138 #define B_CHANNEL_STEREO_BUS 0x2000000      /* + left/right */
00139 #define B_CHANNEL_SURROUND_BUS 0x1000000    /* multichannel */
00140 
00141 /* If you have interactions where some inputs can not be used when some */
00142 /* outputs are used, mark both inputs and outputs with this flag. */
00143 #define B_CHANNEL_INTERACTION 0x80000000UL
00144 /* If input channel #n is simplexed with output channel #n, they should both */
00145 /* have this flag set (different from the previous flag, which is more vague). */
00146 #define B_CHANNEL_SIMPLEX 0x40000000UL
00147 
00148 /* connector values */
00149 /* analog connectors */
00150 #define B_CHANNEL_RCA 0x1
00151 #define B_CHANNEL_XLR 0x2
00152 #define B_CHANNEL_TRS 0x4
00153 #define B_CHANNEL_QUARTER_INCH_MONO 0x8
00154 #define B_CHANNEL_MINI_JACK_STEREO 0x10
00155 #define B_CHANNEL_QUARTER_INCH_STEREO 0x20
00156 #define B_CHANNEL_ANALOG_HEADER 0x100   /* internal on card */
00157 #define B_CHANNEL_SNAKE 0x200           /* or D-sub */
00158 /* digital connectors (stereo) */
00159 #define B_CHANNEL_OPTICAL_SPDIF 0x1000
00160 #define B_CHANNEL_COAX_SPDIF 0x2000
00161 #define B_CHANNEL_COAX_EBU 0x4000
00162 #define B_CHANNEL_XLR_EBU 0x8000
00163 #define B_CHANNEL_TRS_EBU 0x10000
00164 #define B_CHANNEL_SPDIF_HEADER 0x20000  /* internal on card */
00165 /* multi-channel digital connectors */
00166 #define B_CHANNEL_LIGHTPIPE 0x100000
00167 #define B_CHANNEL_TDIF 0x200000
00168 #define B_CHANNEL_FIREWIRE 0x400000
00169 #define B_CHANNEL_USB 0x800000
00170 /* If you have multiple output connectors, only one of which can */
00171 /* be active at a time. */
00172 #define B_CHANNEL_EXCLUSIVE_SELECTION 0x80000000UL
00173 
00174 
00175 typedef enum {
00176     B_MULTI_NO_CHANNEL_KIND,
00177     B_MULTI_OUTPUT_CHANNEL = 0x1,
00178     B_MULTI_INPUT_CHANNEL = 0x2,
00179     B_MULTI_OUTPUT_BUS = 0x4,
00180     B_MULTI_INPUT_BUS = 0x8,
00181     B_MULTI_AUX_BUS = 0x10
00182 } channel_kind;
00183 
00184 struct multi_channel_info {
00185     int32           channel_id;
00186     channel_kind    kind;
00187     uint32          designations;
00188     uint32          connectors;
00189     uint32          _reserved_[4];
00190 };
00191 
00192 
00193 /* Constants */
00194 #define B_MULTI_EVENT_MINMAX        16
00195 /* Event flags/masks */
00196 #define B_MULTI_EVENT_TRANSPORT     0x40000000UL
00197 #define B_MULTI_EVENT_HAS_TIMECODE  0x80000000UL
00198 
00199 /* possible transport events */
00200 #define B_MULTI_EVENT_NONE          0x00000000UL
00201 #define B_MULTI_EVENT_START         0x40010000UL
00202 #define B_MULTI_EVENT_LOCATION      0x40020000UL        /* location when shuttling or locating */
00203 #define B_MULTI_EVENT_SHUTTLING     0x40040000UL
00204 #define B_MULTI_EVENT_STOP          0x40080000UL
00205 #define B_MULTI_EVENT_RECORD        0x40100000UL
00206 #define B_MULTI_EVENT_PAUSE         0x40200000UL
00207 #define B_MULTI_EVENT_RUNNING       0x40400000UL        /* location when running */
00208 /* possible device events */
00209 enum {
00210     B_MULTI_EVENT_STARTED               = 0x1,
00211     B_MULTI_EVENT_STOPPED               = 0x2,
00212     B_MULTI_EVENT_CHANNEL_FORMAT_CHANGED= 0x4,
00213     B_MULTI_EVENT_BUFFER_OVERRUN        = 0x8,
00214     B_MULTI_EVENT_SIGNAL_LOST           = 0x10,
00215     B_MULTI_EVENT_SIGNAL_DETECTED       = 0x20, 
00216     B_MULTI_EVENT_CLOCK_LOST            = 0x40,
00217     B_MULTI_EVENT_CLOCK_DETECTED        = 0x80,
00218     B_MULTI_EVENT_NEW_MODE              = 0x100,
00219     B_MULTI_EVENT_CONTROL_CHANGED       = 0x200
00220 };
00221 
00222 typedef struct multi_get_event_info multi_get_event_info;
00223 struct multi_get_event_info {
00224     size_t      info_size;      /* sizeof(multi_get_event_info) */
00225     uint32      supported_mask; /* what events h/w supports */
00226     uint32      current_mask;   /* current driver value */
00227     uint32      queue_size;     /* current queue size */
00228     uint32      event_count;    /* number of events currently in queue*/
00229     uint32      _reserved[3];
00230 };
00231 
00232 typedef struct multi_set_event_info multi_set_event_info;
00233 struct multi_set_event_info {
00234     size_t      info_size;      /* sizeof(multi_set_event_info) */
00235     uint32      in_mask;        /* what events to wait for */
00236     int32       semaphore;      /* semaphore app will wait on */
00237     uint32      queue_size;     /* minimum number of events to save */
00238     uint32      _reserved[4];
00239 };
00240 
00241 typedef struct multi_get_event multi_get_event;
00242 struct transport_t { /* transport event */
00243     float       out_rate;       /* what rate it's now playing at */
00244     int32       out_hours;      /* location at the time given */
00245     int32       out_minutes;
00246     int32       out_seconds;
00247     int32       out_frames;
00248 };
00249 struct multi_get_event {
00250 
00251 
00252     size_t      info_size;      /* sizeof(multi_get_event) */
00253     uint32      event;
00254     bigtime_t   timestamp;      /* real time at which event was received */
00255     int32       count;          /* used for configuration events */
00256     union {
00257             int32       channels[100];
00258             uint32      clocks;
00259             int32       mode;
00260             int32       controls[100];
00261             struct transport_t transport;
00262             char            _reserved_[400];        
00263     #if defined(__cplusplus)
00264         };
00265     #else
00266         } u;
00267     #endif
00268     uint32  _reserved_1[10];
00269 };
00270 
00271 typedef struct multi_channel_enable multi_channel_enable;
00272 struct multi_channel_enable {
00273     size_t          info_size;          /* sizeof(multi_channel_enable) */
00274     /* this must have bytes for all channels (see multi_description) */
00275     /* channel 0 is lowest bit of first byte */
00276     uchar *         enable_bits;
00277 
00278     uint32          lock_source;
00279     int32           lock_data;
00280     uint32          timecode_source;
00281     uint32 *        connectors;         /* which connector(s) is/are active, per channel */
00282 };
00283 
00284 #include <stdio.h>
00285 
00286 #if defined(__cplusplus)
00287     inline void B_SET_CHANNEL(void * bits, int channel, bool value)
00288     {
00289         ASSERT(channel>=0);
00290         (((uchar *)(bits))[((channel)&0x7fff)>>3] =
00291             (((uchar *)(bits))[((channel)&0x7fff)>>3] & ~(1<<((channel)&0x7))) |
00292             ((value) ? (1<<((channel)&0x7)) : 0));
00293     }
00294     inline bool B_TEST_CHANNEL(const void * bits, int channel)
00295     {
00296         return ((((uchar *)(bits))[((channel)&0x7fff)>>3] >> ((channel)&0x7)) & 1);
00297     }
00298 #else
00299     #define B_SET_CHANNEL(bits, channel, value) \
00300         ASSERT(channel>=0); \
00301         (((uchar *)(bits))[((channel)&0x7fff)>>3] = \
00302             (((uchar *)(bits))[((channel)&0x7fff)>>3] & ~(1<<((channel)&0x7))) | \
00303             ((value) ? (1<<((channel)&0x7)) : 0))
00304     #define B_TEST_CHANNEL(bits, channel) \
00305         ((((uchar *)(bits))[((channel)&0x7fff)>>3] >> ((channel)&0x7)) & 1)
00306 #endif
00307 
00308 typedef struct multi_channel_formats multi_channel_formats;
00309 typedef struct multi_format_info multi_format_info;
00310 typedef struct _multi_format _multi_format;
00311 
00312 struct _multi_format {
00313     uint32          rate;
00314     float           cvsr;
00315     uint32          format;
00316     uint32          _reserved_[3];
00317 };
00318 enum {  /* timecode kinds */
00319     B_MULTI_NO_TIMECODE,
00320     B_MULTI_TIMECODE_30,            /* MIDI */
00321     B_MULTI_TIMECODE_30_DROP_2,     /* NTSC */
00322     B_MULTI_TIMECODE_30_DROP_4,     /* Brazil */
00323     B_MULTI_TIMECODE_25,            /* PAL */
00324     B_MULTI_TIMECODE_24             /* Film */
00325 };
00326 struct multi_format_info {
00327     size_t          info_size;          /* sizeof(multi_format_info) */
00328     bigtime_t       output_latency;
00329     bigtime_t       input_latency;
00330     int32           timecode_kind;
00331     uint32          _reserved_[7];
00332     _multi_format   input;
00333     _multi_format   output;
00334 };
00335 struct multi_channel_formats {
00336     size_t          info_size;          /* sizeof(multi_channel_formats) */
00337     int32           request_channel_count;
00338     int32           request_first_channel;
00339     int32           returned_channel_count;
00340     int32           timecode_kind;
00341     int32           _reserved_[4];
00342     _multi_format *
00343                     channels;
00344     bigtime_t *     latencies;          /* DMA/hardware latencies; client calculates for buffers */
00345 };
00346 
00347 
00348 typedef struct multi_mix_value multi_mix_value;
00349 struct multi_mix_value {
00350     int32           id;
00351     union {
00352         float           gain;
00353         uint32          mux;    /* bitmask of mux points */
00354         bool            enable;
00355         uint32          _reserved_[2];
00356 #if defined(__cplusplus)
00357     };
00358 #else
00359     } u;
00360 #endif
00361     int32           ramp;
00362     uint32          _reserved_2[2];
00363 };
00364 
00365 typedef struct multi_mix_value_info multi_mix_value_info;
00366 struct multi_mix_value_info {
00367     size_t          info_size;      /* sizeof(multi_mix_value_info) */
00368     int32           item_count;
00369     multi_mix_value *
00370                     values;
00371     int32           at_frame;       /* time at which to start the change */
00372 };
00373 
00374 //  only one of these should be set
00375 #define B_MULTI_MIX_JUNCTION 0x1
00376 #define B_MULTI_MIX_GAIN 0x2
00377 #define B_MULTI_MIX_MUX 0x4
00378 #define B_MULTI_MIX_ENABLE 0x8
00379 #define B_MULTI_MIX_KIND_MASK 0xffff
00380 //  any combination of these can be set
00381 #define B_MULTI_MIX_RAMP 0x10000
00382 
00383 typedef struct multi_mix_control multi_mix_control;
00384 
00385 struct gain_t {
00386             float           min_gain;       /* dB */
00387             float           max_gain;       /* dB */
00388             float           granularity;    /* dB */
00389 };
00390 
00391 struct mux_t {
00392     uint32  _reserved;
00393 };
00394 
00395 struct enable_t{
00396     uint32          _reserved;
00397 };
00398 struct multi_mix_control {
00399     int32           id;             /* unique for device -- not same id as any channel/bus ! */
00400     uint32          flags;          /* including kind */
00401     int32           master;         /* or 0 if it's not slaved */
00402     union {
00403         struct gain_t       gain;
00404         struct mux_t        mux;
00405         struct enable_t     enable;
00406         uint32      _reserved[12];
00407 #if defined(__cplusplus)
00408     };
00409 #else
00410     }               u;
00411 #endif
00412     char            name[32];
00413 };
00414 
00415 typedef struct multi_mix_channel_info multi_mix_channel_info;
00416 struct multi_mix_channel_info {
00417     size_t          info_size;      /* sizeof(multi_mix_channel_info) */
00418     int32           channel_count;
00419     int32 *         channels;       /* allocated by caller, lists requested channels */
00420     int32           max_count;      /* in: control ids per channel */
00421     int32           actual_count;   /* out: actual max # controls for any individual requested channel */
00422     int32 **        controls;
00423 };
00424 
00425 typedef struct multi_mix_control_info multi_mix_control_info;
00426 struct multi_mix_control_info {
00427     size_t          info_size;      /* sizeof(multi_mix_control_info) */
00428     int32           control_count;  /* in: number of controls */
00429     multi_mix_control *
00430                     controls;       /* allocated by caller, returns control description for each */
00431 };
00432 
00433 typedef struct multi_mix_connection multi_mix_connection;
00434 struct multi_mix_connection {
00435     int32           from;
00436     int32           to;
00437     uint32          _reserved_[2];
00438 };
00439 
00440 typedef struct multi_mix_connection_info multi_mix_connection_info;
00441 struct multi_mix_connection_info {
00442     size_t          info_size;
00443     int32           max_count;      /* in: available space */
00444     int32           actual_count;   /* out: actual count */
00445     multi_mix_connection *
00446                     connections;    /* allocated by caller, returns connections */
00447 };
00448 
00449 
00450 /* possible flags values for what is available (in and out) */
00451 #define B_MULTI_BUFFER_PLAYBACK         0x00001
00452 #define B_MULTI_BUFFER_RECORD           0x00002
00453 #define B_MULTI_BUFFER_METERING         0x00004
00454 #define B_MULTI_BUFFER_TIMECODE         0x40000
00455 #define B_MULTI_BUFFER_SOFTWARE_SWAP    0x80000
00456 
00457 typedef struct multi_buffer_list multi_buffer_list;
00458 typedef struct buffer_desc buffer_desc;
00459 /* This struct is used to query the driver about what buffers it will use, */
00460 /* and to tell it what buffers to use if it supports soft buffers. */
00461 struct multi_buffer_list {
00462 
00463     size_t          info_size;              /* sizeof(multi_buffer_list) */
00464     uint32          flags;
00465 
00466     int32           request_playback_buffers;
00467     int32           request_playback_channels;
00468     uint32          request_playback_buffer_size;       /* frames per buffer */
00469     int32           return_playback_buffers;            /* playback_buffers[b][] */
00470     int32           return_playback_channels;           /* playback_buffers[][c] */
00471     uint32          return_playback_buffer_size;        /* frames */
00472     buffer_desc **  playback_buffers;
00473     void *          _reserved_1;
00474 
00475     int32           request_record_buffers;
00476     int32           request_record_channels;
00477     uint32          request_record_buffer_size;         /* frames per buffer */
00478     int32           return_record_buffers;
00479     int32           return_record_channels;
00480     uint32          return_record_buffer_size;          /* frames */
00481     buffer_desc **  record_buffers;
00482     void *          _reserved_2;
00483 
00484 };
00485 
00486 struct buffer_desc {
00487     char *          base;       /* pointer to first sample for channel for buffer */
00488     size_t          stride;     /* offset to next sample */
00489     uint32          _reserved_[2];
00490 };
00491 
00492 
00493 /* This struct is used when actually queuing data to be played, and/or */
00494 /* receiving data from a recorder. */
00495 typedef struct multi_buffer_info multi_buffer_info;
00496 struct multi_buffer_info {
00497 
00498     size_t          info_size;          /* sizeof(multi_buffer_info) */
00499     uint32          flags;
00500 
00501     bigtime_t       played_real_time;
00502     bigtime_t       played_frames_count;
00503     int32           _reserved_0;
00504     int32           playback_buffer_cycle;
00505 
00506     bigtime_t       recorded_real_time;
00507     bigtime_t       recorded_frames_count;
00508     int32           _reserved_1;
00509     int32           record_buffer_cycle;
00510 
00511     int32           meter_channel_count;
00512     char *          meters_peak;    /* in the same format as the data; allocated by caller */
00513     char *          meters_average; /* in the same format as the data; allocated by caller */
00514 
00515     /*  timecode sent and received at buffer swap   */
00516     int32           hours;
00517     int32           minutes;
00518     int32           seconds;
00519     int32           tc_frames;      /* for timecode frames as opposed to sample frames */
00520     int32           at_frame_delta; /* how far into buffer (or before buffer for negative) */
00521 
00522 };
00523 
00524 
00525 typedef struct multi_mode_info multi_mode_info;
00526 typedef struct multi_mode_list multi_mode_list;
00527 
00528 struct multi_mode_list {
00529     size_t          info_size;      /* sizeof(multi_mode_list) */
00530     int32           in_request_count;
00531     int32           out_actual_count;
00532     int32           out_current_mode;
00533     multi_mode_info *
00534                     io_modes;
00535 };
00536 
00537 struct multi_mode_info {
00538     int32           mode_id;
00539     uint32          flags;
00540     char            mode_name[64];
00541     int32           input_channel_count;
00542     int32           output_channel_count;
00543     float           best_frame_rate_in;
00544     float           best_frame_rate_out;
00545     uint32          sample_formats_in;
00546     uint32          sample_formats_out;
00547     char            _reserved[160];
00548 };
00549 
00550 
00551 /*  This extension protocol can grow however much you want. */
00552 /*  Good extensions should be put into this header; really  */
00553 /*  good extensions should become part of the regular API.  */
00554 /*  For developer-developed extensions, use all lowercase   */
00555 /*  and digits (no upper case). If we then bless a third-   */
00556 /*  party extension, we can just upper-case the selector.   */
00557 
00558 typedef struct multi_extension_list multi_extension_list;
00559 typedef struct multi_extension_info multi_extension_info;
00560 struct multi_extension_info {
00561     uint32          code;
00562     uint32          flags;
00563     char            name[24];
00564 };
00565 
00566 #define B_MULTI_MAX_EXTENSION_COUNT 31
00567 struct multi_extension_list { /* MULTI_LIST_EXTENSIONS */
00568     size_t          info_size;      /* sizeof(multi_extension_list) */
00569     uint32          max_count;
00570     int32           actual_count;   /* return # of actual extensions */
00571     multi_extension_info *
00572                     extensions;     /* allocated by caller */
00573 };
00574 
00575 typedef struct multi_extension_cmd multi_extension_cmd;
00576 struct multi_extension_cmd {    /* MULTI_GET_EXTENSION and MULTI_SET_EXTENSION */
00577     size_t          info_size;      /* sizeof(multi_extension_cmd) */
00578     uint32          code;
00579     uint32          _reserved_1;
00580     void *          in_data;
00581     size_t          in_size;
00582     void *          out_data;
00583     size_t          out_size;
00584 };
00585 
00586 enum {
00587     B_MULTI_EX_CLOCK_GENERATION = 'CLGE',
00588     B_MULTI_EX_DIGITAL_FORMAT = 'DIFO',
00589     B_MULTI_EX_OUTPUT_NOMINAL = 'OUNO',
00590     B_MULTI_EX_INPUT_NOMINAL = 'INNO'
00591 };
00592 
00593 typedef struct multi_ex_clock_generation multi_ex_clock_generation;
00594 struct multi_ex_clock_generation {
00595     int32           channel;    /* if specific, or -1 for all */
00596     uint32          clock;      /* WORDCLOCK or SUPERCLOCK, typically */
00597 };
00598 
00599 typedef struct multi_ex_digital_format multi_ex_digital_format;
00600 struct multi_ex_digital_format {
00601     int32           channel;    /* if specific, or -1 for all */
00602     uint32          format;     /* B_CHANNEL_*_SPDIF or B_CHANNEL_*_EBU */
00603 };
00604 
00605 enum {
00606     B_MULTI_NOMINAL_MINUS_10 = 1, 
00607     B_MULTI_NOMINAL_PLUS_4
00608 };
00609 
00610 typedef struct multi_ex_nominal_level multi_ex_nominal_level;
00611 struct multi_ex_nominal_level {
00612     int32           channel;    /* if specific, or -1 for all */
00613     int32           level;
00614 };
00615 
00616 #endif /* _MULTI_AUDIO_H */
00617 

